﻿// ECMAScript 5 strict mode
"use strict";

assert2(cr, "cr namespace not created");
assert2(cr.behaviors, "cr.behaviors not created");

/////////////////////////////////////
// Behavior class
cr.behaviors.MouseEvent = function(runtime)
{
	this.runtime = runtime;
	var self = this;
	
	if (!this.runtime.isDomFree)
	{
		jQuery(document).mousemove(
			function(info) {
				self.onMouseMove(info);
			}
		);
			
		jQuery(document).mousedown(
			function(info) {
				self.onMouseDown(info);
			}
		);
		
		jQuery(document).mouseup(
			function(info) {
				self.onMouseUp(info);
			}
		);
	}
	// Use document touch input for fullscreen mode
	var elem = (this.runtime.fullscreen_mode > 0) ? document : this.runtime.canvas;
	
	if (this.runtime.isDirectCanvas)
		elem = window["Canvas"];
	else if (this.runtime.isCocoonJs)
		elem = window;
	
	if (typeof PointerEvent !== "undefined")
	{
		elem.addEventListener("pointerdown",
			function(info) {
				self.onPointerStart(info);
			},
			false
		);
		
		elem.addEventListener("pointermove",
			function(info) {
				self.onPointerMove(info);
			},
			false
		);
		
		elem.addEventListener("pointerup",
			function(info) {
				self.onPointerEnd(info);
			},
			false
		);
		
		// Treat pointer cancellation the same as a touch end
		elem.addEventListener("pointercancel",
			function(info) {
				self.onPointerEnd(info);
			},
			false
		);
	}
	else if (window.navigator["msPointerEnabled"])
	{
		elem.addEventListener("MSPointerDown",
			function(info) {
				self.onPointerStart(info);
			},
			false
		);
		
		elem.addEventListener("MSPointerMove",
			function(info) {
				self.onPointerMove(info);
			},
			false
		);
		
		elem.addEventListener("MSPointerUp",
			function(info) {
				self.onPointerEnd(info);
			},
			false
		);
		
		// Treat pointer cancellation the same as a touch end
		elem.addEventListener("MSPointerCancel",
			function(info) {
				self.onPointerEnd(info);
			},
			false
		);
	}
	else
	{
		elem.addEventListener("touchstart",
			function(info) {
				self.onTouchStart(info);
			},
			false
		);
		
		elem.addEventListener("touchmove",
			function(info) {
				self.onTouchMove(info);
			},
			false
		);
		
		elem.addEventListener("touchend",
			function(info) {
				self.onTouchEnd(info);
			},
			false
		);
		
		elem.addEventListener("touchcancel",
			function(info) {
				self.onTouchEnd(info);
			},
			false
		);
	}
};

(function ()
{
	var behaviorProto = cr.behaviors.MouseEvent.prototype;
	var dummyoffset = {left: 0, top: 0};
	function GetMouseEventBehavior(inst)
	{
		var i, len;
		for (i = 0, len = inst.behavior_insts.length; i < len; i++)
		{
			if (inst.behavior_insts[i] instanceof behaviorProto.Instance)
				return inst.behavior_insts[i];
		}
		
		return null;
	};
	//interactive
	behaviorProto.onMouseMove = function (info)
	{
		var offset = this.runtime.isDomFree ? dummyoffset : jQuery(this.runtime.canvas).offset();
		var x = info.pageX - offset.left;
		var y = info.pageY - offset.top;
		var lx, ly, topx, topy;
		
		var arr = this.my_instances.valuesRef();
		//console.log(arr);
		var i, len, b, inst, topmost = null;
		for (i = 0, len = arr.length; i < len; i++)
		{
			inst = arr[i];
			b = GetMouseEventBehavior(inst);
			
			if (!b.enabled)
				continue;		// don't consider disabled, not-dragging, or dragging by other sources
				
			lx = inst.layer.canvasToLayer(x, y, true);
			ly = inst.layer.canvasToLayer(x, y, false);
			inst.update_bbox();
			if (!inst.contains_pt(lx, ly))
			{
				b.onMouseOuting();
				continue;	
			}
				// don't consider instances not over this point
				
			// First instance found
			if (!topmost)
			{
				topmost = inst;
				topx = lx;
				topy = ly;
				continue;
			}
			
			// Otherwise prefer the topmost instance of all overlapping the point
			if (inst.layer.index > topmost.layer.index)
			{
				topmost = inst;
				topx = lx;
				topy = ly;
				continue;
			}
			
			if (inst.layer.index === topmost.layer.index && inst.get_zindex() > topmost.get_zindex())
			{
				topmost = inst;
				topx = lx;
				topy = ly;
				continue;
			}
			//b.onMouseMoving(lx, ly);
		}
		if (topmost)
			GetMouseEventBehavior(topmost).onMouseMoving(topx, topy);
	};
	
	behaviorProto.onMouseDown = function (info)
	{
		if (info.which !== 1)
			return;		// not left mouse button
			
		this.onInputDown("leftmouse", info.pageX, info.pageY);
	};
	
	behaviorProto.onMouseUp = function (info)
	{
		if (info.which !== 1)
			return;		// not left mouse button
			
		this.onInputUp("leftmouse", info.pageX, info.pageY);
	};
	
	behaviorProto.onTouchStart = function (info)
	{
		if (info.preventDefault && cr.isCanvasInputEvent(info))
			info.preventDefault();
		
		var i, len, t, id;
		for (i = 0, len = info.changedTouches.length; i < len; i++)
		{
			t = info.changedTouches[i];
			
			// directCanvas does not send an identifier
			id = t.identifier;
			this.onInputDown(id ? id.toString() : "<none>", t.pageX, t.pageY);
		}
	};
	
	behaviorProto.onTouchMove = function (info)
	{
		if (info.preventDefault)
			info.preventDefault();
		
		var i, len, t, id;
		for (i = 0, len = info.changedTouches.length; i < len; i++)
		{
			t = info.changedTouches[i];
			id = t.identifier;
			this.onInputMove(id ? id.toString() : "<none>", t.pageX, t.pageY);
		}
	};
	
	behaviorProto.onTouchEnd = function (info)
	{
		if (info.preventDefault && cr.isCanvasInputEvent(info))
			info.preventDefault();
		
		var i, len, t, id;
		for (i = 0, len = info.changedTouches.length; i < len; i++)
		{
			t = info.changedTouches[i];
			id = t.identifier;
			this.onInputUp(id ? id.toString() : "<none>", info.pageX, info.pageY);
		}
	};
	
	behaviorProto.onPointerStart = function (info)
	{
		// Ignore mouse events
		if (info["pointerType"] === info["MSPOINTER_TYPE_MOUSE"] || info["pointerType"] === "mouse")
			return;
			
		if (info.preventDefault && cr.isCanvasInputEvent(info))
			info.preventDefault();
		
		this.onInputDown(info["pointerId"].toString(), info.pageX, info.pageY);
	};
	
	behaviorProto.onPointerMove = function (info)
	{
		// Ignore mouse events
		if (info["pointerType"] === info["MSPOINTER_TYPE_MOUSE"] || info["pointerType"] === "mouse")
			return;
			
		if (info.preventDefault)
			info.preventDefault();
		
		this.onInputMove(info["pointerId"].toString(), info.pageX, info.pageY);
	};
	
	behaviorProto.onPointerEnd = function (info)
	{
		// Ignore mouse events
		if (info["pointerType"] === info["MSPOINTER_TYPE_MOUSE"] || info["pointerType"] === "mouse")
			return;
			
		if (info.preventDefault && cr.isCanvasInputEvent(info))
			info.preventDefault();
		
		this.onInputUp(info["pointerId"].toString(), info.pageX, info.pageY);
	};
	
	behaviorProto.onInputDown = function (src, pageX, pageY)
	{
		var offset = this.runtime.isDomFree ? dummyoffset : jQuery(this.runtime.canvas).offset();
		var x = pageX - offset.left;
		var y = pageY - offset.top;
		var lx, ly, topx, topy;
		
		var arr = this.my_instances.valuesRef();
		
		var i, len, b, inst, topmost = null;
		for (i = 0, len = arr.length; i < len; i++)
		{
			inst = arr[i];
			b = GetMouseEventBehavior(inst);
			
			if (!b.enabled)
				continue;		// don't consider disabled or already-dragging instances
				
			lx = inst.layer.canvasToLayer(x, y, true);
			ly = inst.layer.canvasToLayer(x, y, false);
			inst.update_bbox();
			if (!inst.contains_pt(lx, ly))
				continue;		// don't consider instances not over this point
				
			// First instance found
			if (!topmost)
			{
				topmost = inst;
				topx = lx;
				topy = ly;
				continue;
			}
			
			// Otherwise prefer the topmost instance of all overlapping the point
			if (inst.layer.index > topmost.layer.index)
			{
				topmost = inst;
				topx = lx;
				topy = ly;
				continue;
			}
			
			if (inst.layer.index === topmost.layer.index && inst.get_zindex() > topmost.get_zindex())
			{
				topmost = inst;
				topx = lx;
				topy = ly;
				continue;
			}
		}
		
		if (topmost)
			GetMouseEventBehavior(topmost).onDown(src, topx, topy);
	};
	behaviorProto.onInputMove = function (src, pageX, pageY)
	{
		var offset = this.runtime.isDomFree ? dummyoffset : jQuery(this.runtime.canvas).offset();
		var x = pageX - offset.left;
		var y = pageY - offset.top;
		var lx, ly;
		
		var arr = this.my_instances.valuesRef();
		
		var i, len, b, inst;
		for (i = 0, len = arr.length; i < len; i++)
		{
			inst = arr[i];
			b = GetMouseEventBehavior(inst);
			
			if (!b.enabled || !b.isMouseMoving || (!b.isMouseMoving && b.interativesource !== src))
				continue;		// don't consider disabled, not-dragging, or dragging by other sources
				
			lx = inst.layer.canvasToLayer(x, y, true);
			ly = inst.layer.canvasToLayer(x, y, false);
			b.onMouseMoving(lx, ly);
		}
	};
	behaviorProto.onInputUp = function (src, pageX, pageY)
	{
		var offset = this.runtime.isDomFree ? dummyoffset : jQuery(this.runtime.canvas).offset();
		var x = pageX - offset.left;
		var y = pageY - offset.top;
		var lx, ly, topx, topy;
		
		var arr = this.my_instances.valuesRef();
		
		var i, len, b, inst, topmost = null;
		var isInside = false;
		for (i = 0, len = arr.length; i < len; i++)
		{
			inst = arr[i];
			b = GetMouseEventBehavior(inst);
			if(!b.isMousePressed)
				continue;
			
			if (!b.enabled)
				continue;		// don't consider disabled or already-dragging instances
				
			lx = inst.layer.canvasToLayer(x, y, true);
			ly = inst.layer.canvasToLayer(x, y, false);
			inst.update_bbox();
			
			isInside = inst.contains_pt(lx, ly);
				
			// First instance found
			if (!topmost)
			{
				topmost = inst;
				topx = lx;
				topy = ly;
				continue;
			}
			
			// Otherwise prefer the topmost instance of all overlapping the point
			if (inst.layer.index > topmost.layer.index)
			{
				topmost = inst;
				topx = lx;
				topy = ly;
				continue;
			}
			
			if (inst.layer.index === topmost.layer.index && inst.get_zindex() > topmost.get_zindex())
			{
				topmost = inst;
				topx = lx;
				topy = ly;
				continue;
			}
			
			/* if (b.enabled && b.interativesource === src)
				b.onUp(); */
		}
		if (topmost && isInside)
			GetMouseEventBehavior(topmost).onUp(src, topx, topy);
		if(topmost && !isInside)
			GetMouseEventBehavior(topmost).onUpOutSide(src, topx, topy);
	};
		
	/////////////////////////////////////
	// Behavior type class
	behaviorProto.Type = function(behavior, objtype)
	{
		this.behavior = behavior;
		this.objtype = objtype;
		this.runtime = behavior.runtime;
	};
	
	var behtypeProto = behaviorProto.Type.prototype;

	behtypeProto.onCreate = function()
	{
		
	};
	
	/////////////////////////////////////
	// Behavior instance class
	behaviorProto.Instance = function(type, inst)
	{
		this.type = type;
		this.behavior = type.behavior;
		this.inst = inst;				// associated object instance to modify
		this.runtime = type.runtime;
		//console.log("inst:", inst, "this:",this)
	};
	
	var behinstProto = behaviorProto.Instance.prototype;
	
	behinstProto.onCreate = function()
	{
		this.isMouseOver = false;
		this.isMousePressed =false;
		this.isMouseMoving = false;
		this.mouseCurX = 0;
		this.mouseCurY = 0;
		this.mousePreX = 0;
		this.mousePreY = 0;
		this.dx = 0;
		this.dy = 0;
		this.interativesource = "<none>";
		this.enabled = (this.properties[0] !== 0);
	};
	behinstProto.onMouseOuting = function ()
	{
		this.isMouseMoving = false;
		if(this.isMouseOver)
			this.runtime.trigger(cr.behaviors.MouseEvent.prototype.cnds.OnMouseLeave, this.inst);
	}
	behinstProto.onMouseMoving = function(x,y)
	{
		
		if(this.inst!=null)
		{
			if(Math.round(x)==Math.round(this.mousePreX) && Math.round(y) ==Math.round(this.mousePreY))
				this.isMouseMoving = false;
			else
				this.isMouseMoving = true;
			
			this.mousePreX = this.mouseCurX;
			this.mousePreY = this.mouseCurY;
			
			this.mouseCurX = x;
			this.mouseCurY = y;
			if (!this.isMouseOver)
			{
				this.runtime.trigger(cr.behaviors.MouseEvent.prototype.cnds.OnMouseOver, this.inst);
			}
		}
	};
	
	behinstProto.onDown = function(src, x, y)
	{
		this.isMousePressed = true;
		this.dx = x - this.inst.x;
		this.dy = y - this.inst.y;
		this.interativesource = src;
		
		this.runtime.isInUserInputEvent = true;
		this.runtime.trigger(cr.behaviors.MouseEvent.prototype.cnds.OnMouseDown, this.inst);
		this.runtime.isInUserInputEvent = false;
	};
	
	behinstProto.onUp = function()
	{
		this.isMousePressed = false;
		this.runtime.isInUserInputEvent = true;
		this.runtime.trigger(cr.behaviors.MouseEvent.prototype.cnds.OnMouseUp, this.inst);
		this.runtime.isInUserInputEvent = false;
	};
	
	behinstProto.onUpOutSide = function()
	{
		this.isMousePressed = false;
		this.runtime.isInUserInputEvent = true;
		this.runtime.trigger(cr.behaviors.MouseEvent.prototype.cnds.OnMouseUpOutSide, this.inst);
		this.runtime.isInUserInputEvent = false;
	};
	
	behinstProto.saveToJSON = function ()
	{
		return {
			"enabled": this.enabled
		};
	};
	
	behinstProto.loadFromJSON = function (o)
	{
		this.enabled = o["enabled"];
	};
		
	behinstProto.tick = function ()
	{
		//console.log(this, this.inst,"tick")
		/*var dt = this.runtime.getDt(this.inst);
		
		if (dt === 0)
			return;
		
		}*/
	};
	
	/**BEGIN-PREVIEWONLY**/
	behinstProto.getDebuggerValues = function (propsections)
	{
		propsections.push({
			"title": this.type.name,
			"properties": [
				{"name": "Enabled", "value": !!this.enabled}
			]
		});
	};
	
	behinstProto.onDebugValueEdited = function (header, name, value)
	{
		if (name === "Enabled")
			this.enabled = value;
	};
	/**END-PREVIEWONLY**/

	//////////////////////////////////////
	// Conditions
	function Cnds() {};
	Cnds.prototype.OnMouseDown = function()
	{
		//console.log("mouseDown");
		return true;
	};
	Cnds.prototype.OnMouseUp = function()
	{
		//console.log("mouseUp");
		return true;
	};
	
	Cnds.prototype.OnMouseUpOutSide = function()
	{
		//console.log("mouseUp");
		return true;
	};
	
	Cnds.prototype.OnMouseMoving = function()
	{
		//console.log("mouseUp");
		return this.isMouseMoving;
	};
	
	Cnds.prototype.OnMouseOver = function()
	{
		this.isMouseOver = true;
		
		//console.log("mouseOver");
		return true;
	};
	
	Cnds.prototype.OnMouseLeave = function()
	{
		this.isMouseOver = false;
		//console.log("mouseOut");
		return true;
	};
	
	Cnds.prototype.IsEnabled = function ()
	{
		return !! this.enabled;
	};
	
	behaviorProto.cnds = new Cnds();

	//////////////////////////////////////
	// Actions
	function Acts() {};

	Acts.prototype.SetEnabled = function (s)
	{
		this.enabled = (s !== 0);
		
		// Got disabled: cancel any drag
		if (!this.enabled)
			this.isMouseOver = false;
	};
	
	behaviorProto.acts = new Acts();

	//////////////////////////////////////
	// Expressions
	function Exps() {};
	behaviorProto.exps = new Exps();
	
}());